function [share] = LUCM2_project_land_use_change(Coeff_table, co2effect, scrubeffect, allocation, areas, price, Co2price, user_last_year, share_other )
%
% Land Use Change Module - function 2 (LUCM2)
%
% This file estimates future land use shares, according to the
% specification given by Olssen and Kerr (2011).
%

% Code by Simon Anastasiadis : 2011-11-01

%% Parameters

% first year of simulation
start_year = 2009;

% linearization year
linear_year = 2019;

% column numbers
c.year = 1;
c.dairy = 2;
c.sheep = 3;
c.forest = 4;
c.scrub = 5;

%% Limit simulation to 2050
% extending the simulation beyond 2050 requires ensuring that the results
% are sensible, changing the code here and changing the data for the LUCM.

% if last_year exceeds 2050 make it 2050
last_year = min(user_last_year,2050);

% require last_year to exceed 2020
% if final_year < 2020 handle at end
last_year = max(last_year,2020);

%% Trim Prices
% to match the simulation years

% indicator
iyear = last_year >= price.year & price.year>=start_year-1;
% trimming
price.year     = price.year(iyear);
price.interest = price.interest(iyear);
% we used lag prices so lag the indicator
iyear = [iyear(2:end) ; false];
% trimming
price.dairy    = price.dairy(iyear);
price.sheep    = price.sheep(iyear);
price.forest   = price.forest(iyear);

%% Set allocation to match simulation years

% years between start_year and first allocation.year
yr_dif = allocation.year(1) - start_year + 1;

% extend allocation back to start_year
allocation.dairyfree  = [ones(yr_dif,1);allocation.dairyfree];
allocation.dairy2for1 = [ones(yr_dif,1);allocation.dairy2for1];
allocation.sheepfree  = [ones(yr_dif,1);allocation.sheepfree];
allocation.sheep2for1 = [ones(yr_dif,1);allocation.sheep2for1];

% extend years back to start_year
allocation.year = [((start_year-1):(start_year+yr_dif-2)).';allocation.year];

% indicator
iyear = last_year >= allocation.year & allocation.year >= start_year-1;
% trim allocation to match simulation years
allocation.year       = allocation.year(iyear);
allocation.dairyfree  = allocation.dairyfree(iyear);
allocation.dairy2for1 = allocation.dairy2for1(iyear);
allocation.sheepfree  = allocation.sheepfree(iyear);
allocation.sheep2for1 = allocation.sheep2for1(iyear);

%% Data structure note:
% At this point we switch from using structures to store the data (where we
% refer to area.dairy) and switch instead to using matrices to store the
% data (where we refer to share(:,2)).
%
% See also the parameter: column numbers.
%
% The use of matrices instead of structures to store data also occurs in
% the subfunction: predict_shares.
% This subfunction follows the same convention as the parameter: column
% numbers.
%

%% Baseline estimates

base_shares = predict_shares(Coeff_table, price, share_other, areas);

%% Co2 incorporated estimates

% modify price
adj_price = adjust_prices(price, co2effect, scrubeffect, allocation, Co2price);
% estimate land use shares
ETS_shares = predict_shares(Coeff_table, adj_price, share_other, areas);

%% Dairy adjusted estimates

% modify price
adjD_price = adjustD_prices(price, co2effect, scrubeffect, allocation, Co2price);
% estimate land use shares
adjD_shares = predict_shares(Coeff_table, adjD_price, share_other, areas);

%% Linearize over first 10 years

% setup share
share = zeros(size(ETS_shares));
share(:,c.year) = ETS_shares(:,c.year);
% indicator and duration
linear_t = linear_year - start_year + 1;
ishare = share(:,c.year)==linear_year;

% Dairy share
share(:,c.dairy) = adjD_shares(:,c.dairy);
% step size
d_share = (adjD_shares(ishare,c.dairy) - base_shares(ishare,c.dairy)) / linear_t;
% linearize
share(2:linear_t,c.dairy) = base_shares(2:11,c.dairy) + (1:linear_t-1).' * d_share;

% Sheep & Beef share
share(:,c.sheep) = ETS_shares(:,c.sheep);
% step size
d_share = (ETS_shares(ishare,c.sheep) - base_shares(ishare,c.sheep)) / linear_t;
% linearize
share(2:linear_t,c.sheep) = base_shares(2:11,c.sheep) + (1:linear_t-1).' * d_share;

% Forestry share
share(:,c.forest) = ETS_shares(:,c.forest);
% step_size
d_share = (ETS_shares(ishare,c.forest) - base_shares(ishare,c.forest)) / linear_t;
% linearize
share(2:linear_t,c.forest) = base_shares(2:11,c.forest) + (1:linear_t-1).' * d_share;

% Scrub share
share(:,c.scrub) = ETS_shares(:,c.scrub);
% step_size
d_share = (ETS_shares(ishare,c.scrub) - base_shares(ishare,c.scrub)) / linear_t;
% linearize
share(2:linear_t,c.scrub) = base_shares(2:11,c.scrub) + (1:linear_t-1).' * d_share;

%% Dairy adjustment

% different in dairy share
d_dairy = sum(ETS_shares(1,c.dairy:c.scrub)) - sum(share(:,2:5),c.dairy);
% spread difference evenly
share(:,c.sheep:c.scrub) = share(:,c.sheep:c.scrub) + d_dairy * ones(1,3) / 3;

%% Trim share is user_last_year < 2020

ishare = share(:,c.year) <= user_last_year;
share = share(ishare,:);

%% Inform User projecting land use change is complete

msg = sprintf(' - - LUCM2 : project land use change complete');
disp(msg)

end

%% Subfunction : Predict Land Use Shares

function [shares] = predict_shares(Coeff, price, share_other, areas)

% log prices (and interst and other_share) setup
log_prices = zeros(size(price.year,1),6);
% log prices
log_prices(:,1) = price.year;
log_prices(:,2) = log(price.dairy);
log_prices(:,3) = log(price.sheep);
log_prices(:,4) = log(price.forest);
log_prices(:,5) = share_other;
log_prices(:,6) = price.interest;

% difference of log prices
dlog_prices = log_prices(2:end,:) - log_prices(1:(end-1),:);

% land use shares setup
shares = zeros(size(log_prices,1),5);
shares(:,1) = log_prices(:,1);
% initial share
initial_LU = [areas.dairy(end), areas.sheep(end), areas.forest(end), areas.scrub(end)];
shares(1,2:end) = initial_LU / sum(initial_LU) * (1 - share_other);

% loop through simulation years and do the estimation
for ii = 1:size(dlog_prices,1)
    % long run dairy share
    LR_dairy = log_prices(ii,2:end) * Coeff(10:14,1) + Coeff(1,1);
    % long run sheepbeef share
    LR_sheep = log_prices(ii,2:end) * Coeff(10:14,2) + Coeff(1,2);
    % long run forestry share
    LR_forest= log_prices(ii,2:end) * Coeff(10:14,3) + Coeff(1,3);
    
    % calculate change in dairy share
    dshare_dairy = dlog_prices(ii,2:end) * Coeff(2:6,1) ...
                    - Coeff(7,1) * (shares(ii,2) - LR_dairy) ...
                    - Coeff(8,1) * (shares(ii,3) - LR_sheep) ...
                    - Coeff(9,1) * (shares(ii,4) - LR_forest);
    % calculate change in sheepbeef share
    dshare_sheep = dlog_prices(ii,2:end) * Coeff(2:6,2) ...
                    - Coeff(7,2) * (shares(ii,2) - LR_dairy) ...
                    - Coeff(8,2) * (shares(ii,3) - LR_sheep) ...
                    - Coeff(9,2) * (shares(ii,4) - LR_forest);
    % calculate change in forestry share
    dshare_forest = dlog_prices(ii,2:end) * Coeff(2:6,3) ...
                    - Coeff(7,3) * (shares(ii,2) - LR_dairy) ...
                    - Coeff(8,3) * (shares(ii,3) - LR_sheep) ...
                    - Coeff(9,3) * (shares(ii,4) - LR_forest);
    % calculate change in scrub share
    dshare_scrub = dlog_prices(ii,2:end) * Coeff(2:6,4) ...
                    - Coeff(7,4) * (shares(ii,2) - LR_dairy) ...
                    - Coeff(8,4) * (shares(ii,3) - LR_sheep) ...
                    - Coeff(9,4) * (shares(ii,4) - LR_forest);
    
    % share changes
    d_shares = [dshare_dairy, dshare_sheep, dshare_forest, dshare_scrub];
    % calculate new shares
    shares(ii+1,2:end) = shares(ii,2:end) + d_shares;
end

end

%% Subfunction : Adjust Prices for ETS

function [adj_price] = adjust_prices(price, co2effect, scrubeffect, allocation, Co2price)

% convert Co2price from dollars to cents
Co2price = 100 * Co2price;

% adjust dairy price
% scrub effect
price.dairy = price.dairy - scrubeffect.dairy * Co2price;
% direct ETS effect
price.dairy = price.dairy + co2effect.dairy .* Co2price ...
              .* (1 - allocation.dairyfree) .* (1 - allocation.dairy2for1);

% adjust sheep price
% scrub effect
price.sheep = price.sheep - scrubeffect.sheep * Co2price;
% direct ETS effect
price.sheep = price.sheep + co2effect.sheep .* Co2price ...
              .* (1 - allocation.sheepfree) .* (1 - allocation.sheep2for1);

% adjust forest price
% scrub effect
price.forest = price.forest - scrubeffect.forest * Co2price;
% direct ETS effect
price.forest = price.forest + co2effect.forest * Co2price;

% make price = adj_price
adj_price = price;

end

%% Subfunction : Adjust Prices for ETS - dairy modification

function [adjD_price] = adjustD_prices(price, co2effect, scrubeffect, allocation, Co2price)

% convert Co2price from dollars to cents
Co2price = 100 * Co2price;

% adjust dairy price
% scrub effect
price.dairy = price.dairy - scrubeffect.dairy * Co2price;
% direct ETS effect
price.dairy = price.dairy + co2effect.dairy .* Co2price ...
              .* (1 - allocation.dairyfree) .* (1 - allocation.dairy2for1);

% adjust sheep price
% scrub effect
price.sheep = price.sheep - scrubeffect.sheep * Co2price;
% direct ETS effect
price.sheep = price.sheep + co2effect.sheep .* Co2price ...
              .* (1 - allocation.sheepfree) .* (1 - allocation.sheep2for1);

% Due the interaction between forestry price and dairy share we exclude the
% change in forestry price due to ETS to better estimate dairy share.

% make price = adj_price
adjD_price = price;

end

